// BMI Interactive Assessment JavaScript
// This script handles all interactive functionality including slider controls,
// real-time BMI calculations, visual feedback, and human figure updates

class BMICalculator {
    constructor() {
        // Initialize DOM elements
        this.massSlider = document.getElementById('mass-slider');
        this.heightSlider = document.getElementById('height-slider');
        this.massValue = document.getElementById('mass-value');
        this.heightValue = document.getElementById('height-value');
        this.heightCm = document.getElementById('height-cm');
        this.bmiNumber = document.getElementById('bmi-number');
        this.bmiCategory = document.getElementById('bmi-category');
        this.bmiOutput = document.getElementById('bmi-output');
        this.formulaDisplay = document.getElementById('formula-display');
        this.resetBtn = document.getElementById('reset-btn');
        this.hintsBtn = document.getElementById('hints-btn');
        this.hintsModal = document.getElementById('hints-modal');
        this.humanFigure = document.getElementById('human-figure');
        
        // Initialize human body parts for scaling
        this.bodyParts = {
            head: document.getElementById('head'),
            torso: document.getElementById('torso'),
            leftArm: document.getElementById('left-arm'),
            rightArm: document.getElementById('right-arm'),
            leftLeg: document.getElementById('left-leg'),
            rightLeg: document.getElementById('right-leg'),
            leftFoot: document.getElementById('left-foot'),
            rightFoot: document.getElementById('right-foot')
        };
        
        // Default values
        this.defaultMass = 60;
        this.defaultHeight = 1.65;
        
        // Initialize the application
        this.init();
    }
    
    init() {
        // Set up event listeners for sliders
        this.massSlider.addEventListener('input', () => this.updateBMI());
        this.heightSlider.addEventListener('input', () => this.updateBMI());
        
        // Button event listeners
        this.resetBtn.addEventListener('click', () => this.resetValues());
        this.hintsBtn.addEventListener('click', () => this.showHints());
        
        // Modal close functionality
        const closeBtn = document.querySelector('.close');
        closeBtn.addEventListener('click', () => this.hideHints());
        window.addEventListener('click', (e) => {
            if (e.target === this.hintsModal) {
                this.hideHints();
            }
        });
        
        // Initial calculation and display
        this.updateBMI();
        
        // Add keyboard support for accessibility
        this.addKeyboardSupport();
    }
    
    // Calculate BMI using the standard formula: BMI = mass(kg) / height(m)²
    calculateBMI(mass, height) {
        return mass / (height * height);
    }
    
    // Update all displays when sliders change
    updateBMI() {
        // Get current values from sliders
        const mass = parseFloat(this.massSlider.value);
        const height = parseFloat(this.heightSlider.value);
        
        // Update display values
        this.massValue.textContent = mass;
        this.heightValue.textContent = height.toFixed(2);
        this.heightCm.textContent = Math.round(height * 100);
        
        // Calculate BMI
        const bmi = this.calculateBMI(mass, height);
        this.bmiNumber.textContent = bmi.toFixed(1);
        
        // Update formula display with current values
        this.formulaDisplay.textContent = 
            `BMI = ${mass} ÷ (${height.toFixed(2)} × ${height.toFixed(2)}) = ${bmi.toFixed(1)}`;
        
        // Determine and display BMI category
        const category = this.getBMICategory(bmi);
        this.bmiCategory.textContent = category.name;
        
        // Update visual styling based on category
        this.updateBMIVisuals(category);
        
        // Update human figure to reflect current mass and height
        this.updateHumanFigure(mass, height, bmi);
    }
    
    // Determine BMI category based on standard ranges
    getBMICategory(bmi) {
        if (bmi < 18.5) {
            return { name: 'Underweight', class: 'underweight', color: '#2196f3' };
        } else if (bmi >= 18.5 && bmi < 25) {
            return { name: 'Healthy Weight', class: 'healthy', color: '#4caf50' };
        } else if (bmi >= 25 && bmi < 30) {
            return { name: 'Overweight', class: 'overweight', color: '#ff9800' };
        } else {
            return { name: 'Obese', class: 'obese', color: '#f44336' };
        }
    }
    
    // Update visual styling of BMI output based on category
    updateBMIVisuals(category) {
        // Remove all category classes
        this.bmiOutput.className = 'bmi-output';
        // Add current category class
        this.bmiOutput.classList.add(category.class);
    }
    
    // Update human figure to reflect current mass and height values
    // This function implements proportional scaling for anatomical accuracy
    updateHumanFigure(mass, height, bmi) {
        // Calculate scaling factors based on ranges
        // Height scaling: 1.3m to 2.0m maps to 0.85 to 1.15 scale
        const heightScale = 0.85 + ((height - 1.3) / (2.0 - 1.3)) * 0.3;
        
        // Mass scaling affects width/thickness: 30kg to 120kg maps to 0.7 to 1.4 scale
        const massScale = 0.7 + ((mass - 30) / (120 - 30)) * 0.7;
        
        // BMI-based adjustments for more realistic body representation
        const bmiScale = Math.max(0.8, Math.min(1.3, bmi / 22)); // Normalize around BMI 22
        
        // Update SVG viewBox for overall height scaling
        const baseViewBox = 300;
        const scaledViewBox = baseViewBox / heightScale;
        this.humanFigure.setAttribute('viewBox', `0 0 200 ${scaledViewBox}`);
        
        // Scale individual body parts with anatomical proportions
        this.scaleBodyPart(this.bodyParts.head, heightScale * 0.9, massScale * 0.8, 100, 40);
        this.scaleBodyPart(this.bodyParts.torso, heightScale, massScale, 100, 105);
        this.scaleBodyPart(this.bodyParts.leftArm, heightScale, massScale * 0.9, 65, 105);
        this.scaleBodyPart(this.bodyParts.rightArm, heightScale, massScale * 0.9, 135, 105);
        this.scaleBodyPart(this.bodyParts.leftLeg, heightScale, massScale * 0.9, 89, 185);
        this.scaleBodyPart(this.bodyParts.rightLeg, heightScale, massScale * 0.9, 111, 185);
        
        // Adjust feet position based on leg scaling
        const footY = 235 * heightScale;
        this.bodyParts.leftFoot.setAttribute('cy', footY);
        this.bodyParts.rightFoot.setAttribute('cy', footY);
        this.bodyParts.leftFoot.setAttribute('rx', 12 * massScale * 0.8);
        this.bodyParts.rightFoot.setAttribute('rx', 12 * massScale * 0.8);
    }
    
    // Helper function to scale individual body parts proportionally
    scaleBodyPart(element, heightScale, widthScale, centerX, centerY) {
        if (!element) return;
        
        // Apply transform for proportional scaling around the center point
        const transform = `translate(${centerX}, ${centerY}) scale(${widthScale}, ${heightScale}) translate(${-centerX}, ${-centerY})`;
        element.setAttribute('transform', transform);
    }
    
    // Reset all values to defaults
    resetValues() {
        this.massSlider.value = this.defaultMass;
        this.heightSlider.value = this.defaultHeight;
        this.updateBMI();
        
        // Add visual feedback for reset action
        this.resetBtn.style.background = '#27ae60';
        this.resetBtn.textContent = 'Reset ✓';
        
        setTimeout(() => {
            this.resetBtn.style.background = '#4a90e2';
            this.resetBtn.textContent = 'Reset Values';
        }, 1000);
    }
    
    // Show hints modal
    showHints() {
        this.hintsModal.style.display = 'block';
        // Focus management for accessibility
        this.hintsModal.querySelector('.close').focus();
    }
    
    // Hide hints modal
    hideHints() {
        this.hintsModal.style.display = 'none';
        // Return focus to hints button
        this.hintsBtn.focus();
    }
    
    // Add keyboard support for better accessibility
    addKeyboardSupport() {
        // Allow keyboard navigation for sliders
        [this.massSlider, this.heightSlider].forEach(slider => {
            slider.addEventListener('keydown', (e) => {
                let step = parseFloat(slider.step);
                let currentValue = parseFloat(slider.value);
                let newValue = currentValue;
                
                switch(e.key) {
                    case 'ArrowUp':
                    case 'ArrowRight':
                        newValue = Math.min(parseFloat(slider.max), currentValue + step);
                        break;
                    case 'ArrowDown':
                    case 'ArrowLeft':
                        newValue = Math.max(parseFloat(slider.min), currentValue - step);
                        break;
                    case 'Home':
                        newValue = parseFloat(slider.min);
                        break;
                    case 'End':
                        newValue = parseFloat(slider.max);
                        break;
                    default:
                        return; // Don't prevent default for other keys
                }
                
                e.preventDefault();
                slider.value = newValue;
                this.updateBMI();
            });
        });
        
        // Escape key to close modal
        document.addEventListener('keydown', (e) => {
            if (e.key === 'Escape' && this.hintsModal.style.display === 'block') {
                this.hideHints();
            }
        });
    }
}

// Initialize the BMI Calculator when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    // Detect if running in iframe for responsive height adjustment
    if (window.self !== window.top) {
        document.body.classList.add('iframe-mode');
    }
    
    // Create new BMI Calculator instance
    const bmiCalc = new BMICalculator();
    
    // Add smooth transitions for better user experience
    document.querySelectorAll('.slider').forEach(slider => {
        slider.addEventListener('input', () => {
            // Add haptic feedback for touch devices
            if ('vibrate' in navigator) {
                navigator.vibrate(10);
            }
        });
    });
    
    // Add tooltip functionality for educational elements
    document.querySelectorAll('[title]').forEach(element => {
        element.addEventListener('mouseenter', (e) => {
            // Custom tooltip could be implemented here for better mobile support
            // For now, relying on native title attribute
        });
    });
    
    console.log('BMI Interactive Assessment loaded successfully');
});